<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class RolePermissionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create Permissions
        $permissions = [
            // Users Management
            ['name' => 'manage_users', 'display_name' => 'Manage Users', 'description' => 'Create, edit, and delete users', 'group' => 'users'],
            ['name' => 'view_users', 'display_name' => 'View Users', 'description' => 'View users list', 'group' => 'users'],

            // Roles & Permissions
            ['name' => 'manage_roles', 'display_name' => 'Manage Roles', 'description' => 'Create, edit, and delete roles', 'group' => 'roles'],
            ['name' => 'view_roles', 'display_name' => 'View Roles', 'description' => 'View roles list', 'group' => 'roles'],
            ['name' => 'manage_permissions', 'display_name' => 'Manage Permissions', 'description' => 'Create, edit, and delete permissions', 'group' => 'roles'],

            // Items Management
            ['name' => 'manage_items', 'display_name' => 'Manage Items', 'description' => 'Create, edit, and delete clothing items', 'group' => 'items'],
            ['name' => 'view_items', 'display_name' => 'View Items', 'description' => 'View items list', 'group' => 'items'],

            // Orders Management
            ['name' => 'manage_orders', 'display_name' => 'Manage Orders', 'description' => 'Create, edit, and delete service orders', 'group' => 'orders'],
            ['name' => 'view_orders', 'display_name' => 'View Orders', 'description' => 'View orders list', 'group' => 'orders'],
            ['name' => 'create_orders', 'display_name' => 'Create Orders', 'description' => 'Create new service orders', 'group' => 'orders'],

            // Transactions
            ['name' => 'manage_transactions', 'display_name' => 'Manage Transactions', 'description' => 'Create, edit, and delete financial transactions', 'group' => 'transactions'],
            ['name' => 'view_transactions', 'display_name' => 'View Transactions', 'description' => 'View transactions list', 'group' => 'transactions'],

            // Cashboxes
            ['name' => 'manage_cashboxes', 'display_name' => 'Manage Cashboxes', 'description' => 'Create, edit, and delete cashboxes', 'group' => 'cashboxes'],
            ['name' => 'view_cashboxes', 'display_name' => 'View Cashboxes', 'description' => 'View cashboxes list', 'group' => 'cashboxes'],

            // Settings
            ['name' => 'manage_settings', 'display_name' => 'Manage Settings', 'description' => 'Update system settings', 'group' => 'settings'],
            ['name' => 'view_settings', 'display_name' => 'View Settings', 'description' => 'View system settings', 'group' => 'settings'],

            // Dashboard
            ['name' => 'view_dashboard', 'display_name' => 'View Dashboard', 'description' => 'Access admin dashboard', 'group' => 'dashboard'],
        ];

        foreach ($permissions as $permission) {
            Permission::firstOrCreate(
                ['name' => $permission['name']],
                $permission
            );
        }

        // Create Roles
        $adminRole = Role::firstOrCreate(
            ['name' => 'admin'],
            [
                'display_name' => 'Administrator',
                'description' => 'Full system access with all permissions'
            ]
        );

        $managerRole = Role::firstOrCreate(
            ['name' => 'manager'],
            [
                'display_name' => 'Manager',
                'description' => 'Can manage orders, items, and transactions'
            ]
        );

        $cashierRole = Role::firstOrCreate(
            ['name' => 'cashier'],
            [
                'display_name' => 'Cashier',
                'description' => 'Can create orders and view basic information'
            ]
        );

        // Assign permissions to roles
        // Admin gets all permissions
        $adminRole->permissions()->sync(Permission::pluck('id'));

        // Manager permissions
        $managerPermissions = Permission::whereIn('name', [
            'view_dashboard',
            'view_items', 'manage_items',
            'view_orders', 'manage_orders', 'create_orders',
            'view_transactions', 'manage_transactions',
            'view_cashboxes', 'manage_cashboxes',
            'view_settings',
        ])->pluck('id');
        $managerRole->permissions()->sync($managerPermissions);

        // Cashier permissions
        $cashierPermissions = Permission::whereIn('name', [
            'view_dashboard',
            'view_items',
            'view_orders', 'create_orders',
            'view_cashboxes',
        ])->pluck('id');
        $cashierRole->permissions()->sync($cashierPermissions);

        // Create default admin user if doesn't exist
        $adminUser = User::firstOrCreate(
            ['email' => 'admin@lavandera.test'],
            [
                'name' => 'Administrator',
                'password' => Hash::make('password'),
                'role_id' => $adminRole->id,
            ]
        );

        // Update existing admin user's role if exists
        if ($adminUser->wasRecentlyCreated === false) {
            $adminUser->update(['role_id' => $adminRole->id]);
        }
    }
}
