<?php

namespace App\Services;

use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class MachineIdService
{
    /**
     * إنشاء Machine ID فريد للجهاز
     */
    public function generateMachineId(): string
    {
        $components = [];
        
        // 1. اسم الجهاز (Hostname)
        if (function_exists('gethostname')) {
            $components[] = gethostname();
        }
        
        // 2. معلومات النظام
        if (PHP_OS_FAMILY === 'Windows') {
            if (function_exists('exec')) {
                $serial = @exec('wmic bios get serialnumber 2>nul');
                if ($serial && $serial !== 'SerialNumber') {
                    $components[] = trim($serial);
                }
            }
        } else {
            if (file_exists('/etc/machine-id')) {
                $components[] = trim(file_get_contents('/etc/machine-id'));
            } elseif (function_exists('exec')) {
                $hostid = @exec('hostid');
                if ($hostid) {
                    $components[] = $hostid;
                }
            }
        }
        
        // 3. معلومات PHP
        $components[] = PHP_VERSION;
        $components[] = PHP_OS;
        
        // 4. معلومات قاعدة البيانات
        $dbName = config('database.connections.mysql.database');
        if ($dbName) {
            $components[] = $dbName;
        }
        
        // 5. APP_KEY (جزء منه)
        $appKey = config('app.key');
        if ($appKey) {
            $components[] = substr($appKey, 0, 16);
        }
        
        // دمج جميع المكونات وإنشاء Hash
        $combined = implode('|', array_filter($components));
        
        if (empty($combined)) {
            $combined = request()->ip() . '|' . request()->userAgent() . '|' . config('app.name');
        }
        
        $machineId = hash('sha256', $combined);
        return substr($machineId, 0, 32);
    }
    
    /**
     * الحصول على Machine ID من Settings أو إنشاؤه
     */
    public function getMachineId(): string
    {
        $machineId = \App\Models\Setting::getValue('machine_id');
        
        if (!$machineId) {
            $machineId = $this->generateMachineId();
            \App\Models\Setting::setValue('machine_id', $machineId);
        }
        
        return $machineId;
    }
    
    /**
     * الحصول على اسم الجهاز
     */
    public function getMachineName(): string
    {
        if (function_exists('gethostname')) {
            return gethostname();
        }
        
        return 'Unknown';
    }
}




