<?php

namespace App\Http\Controllers;

use App\Models\Cashbox;
use App\Models\ClothingItem;
use App\Models\FinancialTransaction;
use App\Models\ServiceOrder;
use App\Models\Setting;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ServiceOrderController extends Controller
{
    public function index()
    {
        $items = ClothingItem::query()
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $settings = [
            'company_name' => Setting::getValue('company_name', 'Lavandera'),
            'receipt_copies' => (int) Setting::getValue('receipt_copies', '1'),
            'currency' => Setting::getValue('currency', 'USD'),
            'barcode_prefix' => Setting::getValue('barcode_prefix', 'MSB'),
        ];

        return view('front.service-orders', compact('items', 'settings'));
    }

    public function searchItems(Request $request): JsonResponse
    {
        $query = $request->get('q', '');
        
        $items = ClothingItem::query()
            ->where('is_active', true)
            ->when($query, function ($q) use ($query) {
                $q->where('name', 'like', '%' . $query . '%');
            })
            ->orderBy('name')
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'name' => $item->name,
                    'image_url' => $item->image_url,
                    'default_wash_price' => $item->default_wash_price,
                    'default_iron_price' => $item->default_iron_price,
                ];
            });

        return response()->json(['items' => $items]);
    }

    public function store(Request $request): JsonResponse
    {
        $data = $request->validate([
            'customer_name' => ['nullable', 'string', 'max:120'],
            'gender' => ['required', 'in:male,female'],
            'service_type' => ['required', 'in:washing,ironing'],
            'discount_amount' => ['nullable', 'numeric', 'min:0'],
            'mark_paid' => ['sometimes', 'boolean'],
            'items' => ['required', 'array', 'min:1'],
            'items.*.id' => ['required', 'exists:clothing_items,id'],
            'items.*.quantity' => ['required', 'integer', 'min:1', 'max:50'],
            'items.*.color' => ['nullable', 'string', 'max:50'],
            'items.*.serviceType' => ['nullable', 'in:washing,ironing'],
        ]);

        $itemsInput = collect($data['items']);
        $itemModels = ClothingItem::whereIn('id', $itemsInput->pluck('id'))->get()->keyBy('id');

        if ($itemModels->count() !== $itemsInput->count()) {
            return response()->json(['message' => 'One or more items are invalid'], 422);
        }

        return DB::transaction(function () use ($data, $itemsInput, $itemModels) {
            $subtotal = 0;

            foreach ($itemsInput as $itemInput) {
                $model = $itemModels[$itemInput['id']];
                // Use item-specific serviceType if provided, otherwise fall back to order service_type
                $itemServiceType = $itemInput['serviceType'] ?? $data['service_type'];
                $unit = $itemServiceType === 'washing'
                    ? $model->default_wash_price
                    : $model->default_iron_price;
                $subtotal += $unit * $itemInput['quantity'];
            }

            $subtotal = round($subtotal, 2);
            $discount = min($subtotal, $data['discount_amount'] ?? 0);
            $total = round($subtotal - $discount, 2);
            $orderNumber = $this->generateOrderNumber();
            $barcode = $orderNumber; // Use order number as barcode

            $cashbox = Cashbox::first();

            $order = ServiceOrder::create([
                'order_number' => $orderNumber,
                'barcode' => $barcode,
                'customer_name' => $data['customer_name'] ?? null,
                'gender' => $data['gender'],
                'service_type' => $data['service_type'],
                'subtotal' => $subtotal,
                'discount_amount' => $discount,
                'total' => $total,
                'status' => 'processing',
                'payment_status' => $data['mark_paid'] ?? false ? 'paid' : 'unpaid',
                'cashbox_id' => $cashbox?->id,
            ]);

            foreach ($itemsInput as $itemInput) {
                $model = $itemModels[$itemInput['id']];
                // Use item-specific serviceType if provided, otherwise fall back to order service_type
                $itemServiceType = $itemInput['serviceType'] ?? $data['service_type'];
                $unit = $itemServiceType === 'washing'
                    ? $model->default_wash_price
                    : $model->default_iron_price;
                $lineTotal = $unit * $itemInput['quantity'];

                $order->items()->create([
                    'clothing_item_id' => $model->id,
                    'quantity' => $itemInput['quantity'],
                    'color' => $itemInput['color'] ?? null,
                    'service_type' => $itemServiceType,
                    'unit_price' => $unit,
                    'line_total' => $lineTotal,
                ]);
            }

            $transaction = null;

            if ($total > 0 && ($data['mark_paid'] ?? false) && $cashbox) {
                $transaction = FinancialTransaction::create([
                    'type' => 'income',
                    'category' => 'laundry-service',
                    'reference_type' => ServiceOrder::class,
                    'reference_id' => $order->id,
                    'cashbox_id' => $cashbox->id,
                    'amount' => $total,
                    'description' => 'Payment received for order ' . $order->order_number,
                    'occurred_on' => now(),
                ]);

                $cashbox->recordEntry($total, 'in', $transaction);
            }

            return response()->json([
                'message' => 'Service order saved successfully.',
                'order_id' => $order->id,
                'receipt_url' => route('service-orders.receipt', [
                    'serviceOrder' => $order->id,
                    'copy' => Setting::getValue('receipt_copies', 1)
                ]),
                'barcode' => $order->barcode,
                'transaction_id' => $transaction?->id,
            ]);
        });
    }

    public function receipt(ServiceOrder $serviceOrder)
    {
        $settings = [
            'company_name' => Setting::getValue('company_name', 'Lavandera'),
            'receipt_copies' => (int) Setting::getValue('receipt_copies', '1'),
            'currency' => Setting::getValue('currency', 'USD'),
            'company_address' => Setting::getValue('company_address', '123 Laundry Street'),
            'company_phone' => Setting::getValue('company_phone', '+000 000 0000'),
        ];

        $serviceOrder->load(['items.clothingItem']);

        return view('front.receipt', compact('serviceOrder', 'settings'));
    }

    public function lookupByBarcode(Request $request): JsonResponse
    {
        $data = $request->validate([
            'barcode' => ['required', 'string'],
        ]);

        $order = ServiceOrder::where('barcode', $data['barcode'])->with('items.clothingItem')->first();

        if (! $order) {
            return response()->json(['message' => 'Order not found'], 404);
        }

        return response()->json([
            'order' => $order,
        ]);
    }

    public function markCollected(ServiceOrder $serviceOrder): JsonResponse
    {
        $serviceOrder->update([
            'status' => 'collected',
            'collected_at' => now(),
        ]);

        return response()->json([
            'message' => 'Order marked as collected.',
        ]);
    }

    protected function generateOrderNumber(): string
    {
        $prefix = Setting::getValue('order_prefix', 'MSB');
        $latestId = ServiceOrder::max('id') + 1;

        return sprintf('%s-%s', strtoupper($prefix), str_pad((string) $latestId, 5, '0', STR_PAD_LEFT));
    }

    protected function generateBarcode(): string
    {
        // Barcode is now the same as order number
        // This method is kept for backward compatibility but is no longer used
        return $this->generateOrderNumber();
    }
}
